import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  Building2, 
  CheckCircle, 
  Phone, 
  Mail,
  Users,
  Ruler,
  Coffee,
  Bath,
  Zap,
  Wind
} from "lucide-react";
import Link from "next/link";

export default function OfficeCabinPage() {
  const officeModels = [
    {
      name: "Standard Office Cabin",
      description: "Basic office solution with attached bathroom",
      dimensions: "6m x 2.4m x 2.8m (H)",
      capacity: "6-8 persons",
      features: ["Office Space", "Attached Bathroom", "Electrical Points", "Windows & Doors", "Vinyl Flooring"]
    },
    {
      name: "Executive Office Cabin",
      description: "Premium office with enhanced features and finishes",
      dimensions: "6m x 3m x 2.8m (H)",
      capacity: "4-6 persons",
      features: ["Executive Interiors", "Premium Finishes", "Air Conditioning Ready", "Kitchenette", "Private Bathroom"]
    },
    {
      name: "Site Office with Pantry",
      description: "Complete site office solution with pantry facility",
      dimensions: "8m x 2.4m x 2.8m (H)",
      capacity: "8-10 persons",
      features: ["Large Office Space", "Pantry Area", "Bathroom", "Storage", "Meeting Area"]
    }
  ];

  const interiorFeatures = [
    {
      icon: <Zap className="h-6 w-6" />,
      title: "Electrical Installation",
      description: "Complete electrical setup with power points, lighting, and switch boards"
    },
    {
      icon: <Wind className="h-6 w-6" />,
      title: "Climate Control Ready",
      description: "Pre-wired for split system air conditioner installation"
    },
    {
      icon: <Coffee className="h-6 w-6" />,
      title: "Pantry & Kitchen",
      description: "Kitchenette with zip hot water system and storage cabinets"
    },
    {
      icon: <Bath className="h-6 w-6" />,
      title: "Attached Bathroom",
      description: "Private bathroom with partition walls and quality fittings"
    }
  ];

  const specifications = [
    "Insulated walls and roof for UAE climate",
    "Fire safety features and emergency exits",
    "Quality doors and windows with locks",
    "Vinyl flooring with non-slip surface",
    "Electrical installation with safety switches",
    "Plumbing ready for water connections",
    "Partition walls for privacy",
    "Ventilation system for air circulation",
    "Weather-resistant exterior coating",
    "Modular design for easy expansion"
  ];

  const useCases = [
    "Construction site offices",
    "Temporary business operations",
    "Project management centers",
    "Administrative facilities",
    "Sales offices and showrooms",
    "Security control centers",
    "Quality control labs",
    "Meeting and conference rooms"
  ];

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <section className="relative py-20 bg-gradient-to-r from-blue-900/80 to-gray-900/80">
        <div className="absolute inset-0 bg-black/40"></div>
        <div className="absolute inset-0">
          <div className="w-full h-full bg-gradient-to-br from-blue-800 via-gray-800 to-slate-800"></div>
        </div>
        
        <div className="relative z-20 container mx-auto px-4 text-center text-white">
          <h1 className="text-4xl md:text-6xl font-bold mb-6">
            Office Porta Cabins
          </h1>
          <p className="text-xl md:text-2xl max-w-3xl mx-auto">
            Professional Office Solutions with Complete Facilities
          </p>
        </div>
      </section>

      {/* Product Overview */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Complete Office Solutions
            </h2>
            <p className="text-lg text-gray-700 leading-relaxed mb-6">
              Our office porta cabins provide fully equipped workspace solutions with attached bathrooms, 
              pantry facilities, and modern amenities. Perfect for construction sites, temporary offices, 
              and business operations across UAE.
            </p>
          </div>

          {/* Office Models Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-16">
            {officeModels.map((model, index) => (
              <Card key={index} className="overflow-hidden hover:shadow-lg transition-shadow">
                <div className="relative h-48 bg-gradient-to-br from-green-800 to-blue-800">
                  <Badge className="absolute top-4 left-4 bg-green-600">
                    Office Solution
                  </Badge>
                </div>
                
                <CardHeader>
                  <CardTitle className="text-xl font-bold">{model.name}</CardTitle>
                  <CardDescription>{model.description}</CardDescription>
                </CardHeader>
                
                <CardContent>
                  <div className="space-y-3 mb-4">
                    <div className="flex items-center text-sm text-gray-600">
                      <Ruler className="h-4 w-4 mr-2" />
                      <span>{model.dimensions}</span>
                    </div>
                    <div className="flex items-center text-sm text-gray-600">
                      <Users className="h-4 w-4 mr-2" />
                      <span>{model.capacity}</span>
                    </div>
                  </div>

                  <div className="mb-4">
                    <h4 className="text-sm font-semibold text-gray-900 mb-2">Included Features:</h4>
                    <div className="space-y-1">
                      {model.features.slice(0, 3).map((feature, idx) => (
                        <div key={idx} className="flex items-center text-sm text-gray-600">
                          <CheckCircle className="h-3 w-3 mr-2 text-green-600" />
                          <span>{feature}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <Button className="w-full bg-green-600 hover:bg-green-700">
                    <Phone className="mr-2 h-4 w-4" />
                    Get Quote
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Interior Features */}
          <div className="mb-16">
            <h2 className="text-3xl font-bold text-gray-900 text-center mb-12">
              Interior Features & Amenities
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
              {interiorFeatures.map((feature, index) => (
                <div key={index} className="text-center">
                  <div className="bg-blue-100 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4">
                    <div className="text-blue-600">
                      {feature.icon}
                    </div>
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 text-sm">
                    {feature.description}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Specifications & Use Cases */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
            {/* Specifications */}
            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6 flex items-center">
                <Building2 className="h-8 w-8 mr-3 text-blue-600" />
                Technical Specifications
              </h2>
              <div className="space-y-3">
                {specifications.map((spec, index) => (
                  <div key={index} className="flex items-start space-x-3">
                    <CheckCircle className="h-5 w-5 text-blue-600 mt-0.5 flex-shrink-0" />
                    <span className="text-gray-700">{spec}</span>
                  </div>
                ))}
              </div>
              
              <Card className="mt-8 bg-blue-50 border-blue-200">
                <CardContent className="p-6">
                  <h3 className="text-lg font-semibold text-blue-900 mb-3">Professional Grade</h3>
                  <p className="text-blue-800 text-sm">
                    Our office cabins are built to professional standards with quality materials 
                    and fittings suitable for business environments and long-term use.
                  </p>
                </CardContent>
              </Card>
            </div>

            {/* Use Cases */}
            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6 flex items-center">
                <Coffee className="h-8 w-8 mr-3 text-green-600" />
                Applications & Use Cases
              </h2>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                {useCases.map((useCase, index) => (
                  <Card key={index} className="p-4 hover:shadow-md transition-shadow">
                    <div className="flex items-center space-x-3">
                      <div className="w-2 h-2 bg-green-600 rounded-full"></div>
                      <span className="text-gray-700 font-medium">{useCase}</span>
                    </div>
                  </Card>
                ))}
              </div>

              <Card className="mt-8 bg-green-50 border-green-200">
                <CardContent className="p-6">
                  <div className="flex items-center mb-3">
                    <Building2 className="h-5 w-5 text-green-600 mr-2" />
                    <h3 className="text-lg font-semibold text-green-900">Customization Available</h3>
                  </div>
                  <p className="text-green-800 text-sm">
                    We can customize office cabins according to your specific requirements including 
                    size modifications, additional features, and special configurations.
                  </p>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-16 bg-green-800 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready for Your Office Solution?
          </h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            Get a professional office porta cabin with complete facilities. Contact us for 
            customized solutions tailored to your business requirements.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-green-800 hover:bg-gray-100 px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call: +971 58 201 2073
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Request Quote
              </Link>
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}